# This is a Python script demonstrating how to extract neural, audio and motion sensor data from a single file in block format
# published on May 8, 2023
import os
import numpy as np
import block_format_functions
import data_types
import motion_sensor_constants as ms_constants
import motion_sensor_functions
import audio_functions

# Set to your file path and file name
folder = ''  # folder with your data files
file = ''  # name of your data file including extension
file_name = os.path.join(folder, file)

# Set the following values from event file
# neural
number_of_adc_bits = 16
is_neural_signed = True
voltage_resolution = 1.95e-7
number_of_channels = 64
neural_sampling_frequency = 32000 #sample rate in Hz
# audio
number_of_audio_bits = 15
audio_sampling_frequency = 100000  # sample rate in Hz
is_audio_signed = True
# motion sensor
accl_max = 2*ms_constants.G  # m/s^2, maximum value of selected range
gyro_max = 250  # degrees per second,  maximum value of selected range
mag_max = ms_constants.MAGNETOMETER_9250_RANGE  # Teslas,  maximum value of selected range. If your logger type is RATLOG-64, use MAGNETOMETER_9150_RANGE, otherwise use MAGNETOMETER_9250_RANGE
number_of_mag_adc_bits = ms_constants.MAGNETOMETER_9250_NUMBER_OF_BITS # if your logger type is RATLOG-64, use MAGNETOMETER_9150_NUMBER_OF_BITS, otherwise use MAGNETOMETER_9250_NUMBER_OF_BITS

# read data from file as bytes
file = open(file_name, "rb")
byte_data = file.read()
file.close()

# check blocks headers are present and report dropped blocks
byte_data = np.frombuffer(byte_data, 'uint8')
block_start_indices, dropped_blocks = block_format_functions.get_block_start_indices(byte_data)

# extract and parse headers
# get timestamps
timestamps = block_format_functions.get_timestamps(block_start_indices, byte_data)


# extract block structure
data_types_present, data_start_indices, data_segment_lengths = block_format_functions.get_partition_data(byte_data)


# Extract neural data
is_neural_present = data_types.DataTypes.NEURAL in data_types_present  #check neural data is present
if is_neural_signed:
    offset = 0
else:
    offset = 2 ** (number_of_adc_bits - 1)
neural_data = []
if is_neural_present:
    neural_bytes = block_format_functions.extract_data_segments(data_types.DataTypes.NEURAL, data_types_present, block_start_indices, byte_data, data_start_indices, data_segment_lengths)
    neural_data_array = block_format_functions.convert_neural_bytes(neural_bytes, voltage_resolution, offset)

    for chanId in range(0, number_of_channels):
        neural_data.append(neural_data_array[chanId:len(neural_data_array):number_of_channels])

# Extract audio data
is_audio_present = data_types.DataTypes.AUDIO in data_types_present
if is_audio_present:
    audio_bytes = block_format_functions.extract_data_segments(data_types.DataTypes.AUDIO, data_types_present,
                    block_start_indices, byte_data, data_start_indices, data_segment_lengths)
    audio_data = audio_functions.convert_audio_data( is_audio_signed, audio_bytes, number_of_audio_bits )


# Extract motion sensor data
is_motion_sensor_present = data_types.DataTypes.MOTIONSENSOR in data_types_present
if is_motion_sensor_present:
    # get motion sensor bytes from file
    motion_sensor_bytes = block_format_functions.extract_data_segments(data_types.DataTypes.MOTIONSENSOR,
                       data_types_present,block_start_indices, byte_data, data_start_indices,
                       data_segment_lengths)
    # convert to uint16
    motion_sensor_ints = np.array(motion_sensor_bytes).view(np.int16)
    # find block starts
    ms_start_block_indices = motion_sensor_functions.extract_motion_sensor_block_starts(motion_sensor_ints,
                                           ms_constants.CONST_ID)
    # get motion sensor timestamp of each block
    motion_sensor_timestamps = motion_sensor_functions.extract_motion_sensor_timestamps(motion_sensor_ints, ms_start_block_indices )
    # sort into accelerometer, gyro and magnetometer
    accelerometer_ints, gyroscope_ints, magnetometer_ints = motion_sensor_functions.extract_motion_sensor(
        motion_sensor_ints, ms_start_block_indices)

    # convert to physical units
    accelerometer_data = motion_sensor_functions.convert_motion_sensor_data(accelerometer_ints,
                       ms_constants.ACCELEROMETER_NUMBER_OF_BITS, accl_max)
    gyroscope_Data = motion_sensor_functions.convert_motion_sensor_data(gyroscope_ints,
                       ms_constants.GYROSCOPE_NUMBER_OF_BITS, gyro_max)
    magnetometer_data = motion_sensor_functions.convert_motion_sensor_data(magnetometer_ints,
                       number_of_mag_adc_bits, mag_max)


    # sort data by axes
    x_accelerometer, y_accelerometer, z_accelerometer = motion_sensor_functions.sort_motion_sensor_by_axes(
        accelerometer_data, ms_constants.NUMBER_OF_AXES)
    x_gyroscope, y_gyroscope, z_gyroscope = motion_sensor_functions.sort_motion_sensor_by_axes(
        gyroscope_Data, ms_constants.NUMBER_OF_AXES)
    x_magnetometer, y_magnetometer, z_magnetometer = motion_sensor_functions.sort_motion_sensor_by_axes(
        magnetometer_data, ms_constants.NUMBER_OF_AXES)

