import numpy as np
import motion_sensor_constants as ms_constants

def extract_motion_sensor_by_type(motion_sensor_ints, ms_start_block_indices, offset_position, segment_length):
    #extracts a single type of motion sensor data each time it is called
    offsets = motion_sensor_ints[ms_start_block_indices +
                                  offset_position]
    segment_sizes = motion_sensor_ints[ms_start_block_indices + segment_length]
    this_type_data = []
    for block_id in range(0, len(ms_start_block_indices)):
        start_idx = ms_start_block_indices[block_id] + offsets[block_id]
        this_block_type_data = motion_sensor_ints[start_idx:start_idx + segment_sizes[block_id]]
        this_type_data.extend(this_block_type_data)
    return this_type_data


def extract_motion_sensor_block_starts(motion_sensor_ints, const_id):
    #find the start of motion sensor blocks
    ms_start_block_indices = []
    for i in range(0, len(motion_sensor_ints)-1):  # extract motion sensor block starts
        if (motion_sensor_ints[i] == const_id[0]) & (motion_sensor_ints[i+1] == const_id[1]):
            ms_start_block_indices.append(i)
    ms_start_block_indices = np.array(ms_start_block_indices)
    return ms_start_block_indices


def extract_motion_sensor(motion_sensor_ints, ms_start_block_indices):
    # extract each type of motion sensor data from motion sensor block
    accelerometer_data = extract_motion_sensor_by_type(motion_sensor_ints, ms_start_block_indices,
                      ms_constants.ACCELEROMETER_OFFSET_POSITION,
                      ms_constants.ACCELEROMETER_LENGTH_POSITION)
    gyroscope_data = extract_motion_sensor_by_type(motion_sensor_ints, ms_start_block_indices,
                      ms_constants.GYROSCOPE_OFFSET_POSITION,
                      ms_constants.GYROSCOPE_LENGTH_POSITION)
    magnetometer_data = extract_motion_sensor_by_type(motion_sensor_ints, ms_start_block_indices,
                              ms_constants.MAGNETOMETER_OFFSET_POSITION,
                              ms_constants.MAGNETOMETER_LENGTH_POSITION)
    return accelerometer_data, gyroscope_data, magnetometer_data


def convert_motion_sensor_data(data, number_of_bits, max_value):
    # scaled motion sensor data
    offset = 2 ** (number_of_bits - 1)
    scaled_data = np.divide(data, float(offset)) * float(max_value)
    return scaled_data


def sort_motion_sensor_by_axes(data, number_of_axes):
    # sort motion sensor data into x y and z axis data
    x_data = data[0::number_of_axes]
    y_data = data[1::number_of_axes]
    z_data = data[2::number_of_axes]
    return x_data, y_data, z_data


def extract_motion_sensor_timestamps(motion_sensor_ints, ms_start_block_indices):
    # motion sensor has its own timestamps (once per block) - read these timestamps
    timestamps = np.empty(np.size(ms_start_block_indices))
    timestamp_indices = ms_start_block_indices + ms_constants.TIMESTAMP_INDEX[0]
    for timestamp_idx in range(0, len(timestamp_indices)):
        this_timestamp = motion_sensor_ints[timestamp_indices[timestamp_idx]:timestamp_indices[timestamp_idx] + 2]
        this_timestamp = np.array(this_timestamp).view(np.uint32) * ms_constants.TIME_RESOLUTION
        timestamps[timestamp_idx] = this_timestamp
    return timestamps
